#include <iostream>
#include "icsneo/icsneocpp.h"
#include <string_view>
#include <vector>
#include <memory>
#include <windows.h>

/* 
* This file performs a TC10 sleep/wake test between 2 TC10 supported devices using libicsneo.
*   Supported devices: RM2_zl
* 
* ToDo: Refactor/cleanup
*/

std::shared_ptr<icsneo::Device> find_device(const char* serial)
{
    for (auto&& dev : icsneo::FindAllDevices()) {
        if (dev->getSerial() == std::string_view(serial)) {
            return dev;
        }
    }
    return nullptr;
}


int main(int argc, const char** argv)
{
    if (argc != 3) {
        std::cout << "usage: " << argv[0] << " <serial_dut_0>" << " <serial_dut_1>" << std::endl;
        return -1;
    }

    /* Look for device 0 - wakeup just incase it's asleep */
    auto device = find_device(argv[1]);
    if (!device) {
        std::cerr << "error: unable to find device 0" << std::endl;
        return -1;
    }
    device->open();
    std::cout << *device << std::endl;
    if (!device->requestTC10Wake(icsneo::Network::NetID::OP_Ethernet1)) {
        std::cerr << "unable to send TC10 Wake Request" << std::endl;
        return -1;
    }
    device->close();

    /* Look for device 1 - wakeup just incase it's asleep */
    device = find_device(argv[2]);
    if (!device) {
        std::cerr << "error: unable to find device 1" << std::endl;
        return -1;
    }
    device->open();
    std::cout << *device << std::endl;
    if (!device->requestTC10Wake(icsneo::Network::NetID::OP_Ethernet1)) {
        std::cerr << "unable to send TC10 Wake Request" << std::endl;
        return -1;
    }
    device->close();

    /* Verify device 0 is awake; request sleep on device 0; then verify both devices are asleep */
    std::cout << "Requesting sleep on device 0..." << std::endl;
    device = find_device(argv[1]);
    device->open();
    auto s1 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s1) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if ((int)s1->sleepStatus) {
        std::cerr << "device " << *device << "is asleep when it should be awake! Aborting test.." << std::endl;
        return -1;
    }
    if (!device->requestTC10Sleep(icsneo::Network::NetID::OP_Ethernet1)) {
        std::cerr << "unable to send TC10 Sleep Request" << std::endl;
        return -1;
    }
    Sleep(100); // give time to sleep
    std::cout << "Verifying both devices are asleep..." << std::endl;
    auto s2 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s2) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!((int)s2->sleepStatus && !(int)s2->wakeStatus)) {
        std::cerr << "device" << *device << " is awake when it should be asleep! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s2->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s2->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    device = find_device(argv[2]);
    device->open();
    auto s3 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s3) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!((int)s3->sleepStatus && !(int)s3->wakeStatus)) {
        std::cerr << "device " << *device << "is awake when it should be asleep! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s3->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s3->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    std::cout << "Sleep Test passed." << std::endl;

    /* Request wake on device 0; then verify both devices are awake */
    std::cout << "Requesting wake on device 0..." << std::endl;
    device = find_device(argv[1]);
    device->open();
    if (!device->requestTC10Wake(icsneo::Network::NetID::OP_Ethernet1)) {
        std::cerr << "unable to send TC10 Wake Request" << std::endl;
        return -1;
    }
    Sleep(100); // give time to wake
    std::cout << "Verifying both devices are awake..." << std::endl;
    auto s4 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s4) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!(!(int)s4->sleepStatus && (int)s4->wakeStatus)) {
        std::cerr << "device " << *device << "is asleep when it should be awake! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s4->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s4->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    device = find_device(argv[2]);
    device->open();
    auto s5 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s5) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!(!(int)s5->sleepStatus && (int)s5->wakeStatus)) {
        std::cerr << "device " << *device << "is awake when it should be asleep! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s5->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s5->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    std::cout << "Wakeup Test passed." << std::endl;


    /* Request sleep on device 1; then verify both devices are asleep */
    std::cout << "Requesting sleep on device 1..." << std::endl;
    device = find_device(argv[2]);
    device->open();
    if (!device->requestTC10Sleep(icsneo::Network::NetID::OP_Ethernet1)) {
        std::cerr << "unable to send TC10 Sleep Request" << std::endl;
        return -1;
    }
    Sleep(100); // give time to sleep
    std::cout << "Verifying both devices are asleep..." << std::endl;
    auto s6 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s6) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!((int)s6->sleepStatus && !(int)s6->wakeStatus)) {
        std::cerr << "device " << *device << "is awake when it should be asleep! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s6->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s6->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    device = find_device(argv[1]);
    device->open();
    auto s7 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s7) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!((int)s7->sleepStatus && !(int)s7->wakeStatus)) {
        std::cerr << "device " << *device << "is awake when it should be asleep! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s7->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s7->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    std::cout << "Sleep Test passed." << std::endl;

    /* Request wake on device 1; then verify both devices are awake */
    std::cout << "Requesting wake on device 1..." << std::endl;
    device = find_device(argv[2]);
    device->open();
    if (!device->requestTC10Wake(icsneo::Network::NetID::OP_Ethernet1)) {
        std::cerr << "unable to send TC10 Wake Request" << std::endl;
        return -1;
    }
    Sleep(100); // give time to wake
    std::cout << "Verifying both devices are awake..." << std::endl;
    auto s8 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s8) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!(!(int)s8->sleepStatus && (int)s8->wakeStatus)) {
        std::cerr << "device " << *device << "is asleep when it should be awake! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s8->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s8->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    device = find_device(argv[1]);
    device->open();
    auto s9 = device->getTC10Status(icsneo::Network::NetID::OP_Ethernet1);
    if (!s9) {
        std::cerr << "unable to get TC10 status" << std::endl;
        return -1;
    }
    if (!(!(int)s9->sleepStatus && (int)s9->wakeStatus)) {
        std::cerr << "device " << *device << "is awake when it should be asleep! Aborting test.." << std::endl;
        std::cout << "wake status: " << (int)s9->wakeStatus << std::endl;
        std::cout << "sleep status: " << (int)s9->sleepStatus << std::endl;
        return -1;
    }
    device->close();
    std::cout << "Wakeup Test passed." << std::endl;

    std::cout << "Tests finished successfully! Exiting..." << std::endl;

    return 0;
}