from util.util import (
    MyArgParseHelpFormatter,
    argparse_raw,
    open_device,
    add_python_ics_extensions,
)

from util.util_macsec import (
    macsec_clear,
    macsec_load_from_yaml,
)

import ics
import yaml
import argparse

###############################################################################
# This loads a single macsec configuration from selected yaml into supported
#  devices and exits
###############################################################################


DESCRIPTION = "MACSec utility"


def main():
    parser = get_parser()
    args = parser.parse_args()
    add_python_ics_extensions()
    run_test(args, None)


def validate_port_input(port):
    def get_value(port, key, default):
        if key in port:
            return port[key]
        else:
            return default

    inputs = {}
    # using device network
    # default values
    inputs["config_netid"] = get_value(port, "config_netid", "NETID_I2C2")
    inputs["netid"] = get_value(port, "netid", "NETID_OP_ETHERNET1")
    return inputs


def run_test(args, power_supply):
    port = validate_port_input(args.port)
    print(f"Attempting to load {args.yaml}...")

    fail = 0

    # connect to test device1 and set network settings for the macsec test
    with open_device(args.serial_number) as device:
        # Just load default every time in case we get some crazy stuff
        # ics.load_default_settings(device)

        # if we only want reset, exit here
        if args.reset:
            macsec_clear(device, port["config_netid"], port["netid"])
            exit(0)

        # load yaml macsec config
        macsec_load_from_yaml(device, port["config_netid"], port["netid"], args.yaml)

        if fail:  # bail out if something fails
            print(f"{fail} tests failed.")
            exit(1)


def get_parser():
    parser = argparse.ArgumentParser(
        description="MACsec utility script",
        formatter_class=MyArgParseHelpFormatter,
    )
    parser.add_argument(
        "serial_number",
        help="The serial number of the device to connect to",
    )
    parser.add_argument(
        "--yaml",
        help="yaml file name to be loaded/configured.",
        default="test_macsec_Rule_wildcard_SecY_128_strip_strict_sci_SA_sak0.yml",
    )
    parser.add_argument(
        "--port",
        help=argparse_raw(
            "Ethernet port under test. Options in YAML format.\n"
            + "Options include:\n"
            + "'config_netid' - network ID for the port configuration network. Default NETID_I2C2.\n"
            + "'netid' - network ID for the port network. Default NETID_OP_ETHERNET1.\n"
        ),
        required=True,
        type=yaml.safe_load,
    )
    parser.add_argument(
        "--reset",
        help="Only reset and clear and disable macsec on the device, then exit.",
        default=False,
        action="store_true",
    )
    return parser


if __name__ == "__main__":
    main()
