from util.util_exit import ExitCode
from contextlib import contextmanager
import ics
import time
import argparse
import ctypes


def add_python_ics_extensions():
    """
    This will add extensions to the python_ics module.
    Some functions are implemented on the Intrepid DLL, but not included in python_ics.
    These functions can be injected here.
    """

    def icsneoWriteMACsecConfig(device, macsec, netid):
        dll = ctypes.windll.LoadLibrary(str(ics.get_library_path()))
        fileDataCType = (ctypes.c_char * len(macsec))(
            *macsec
        )  # take python fileData[] and convert to char array
        fileDataSizeCType = (ctypes.c_size_t)(
            len(fileDataCType)
        )  # create fileData size as size_t type
        netidCType = (ctypes.c_uint)(netid)  # convert python phyIndx to char
        f = dll[256]
        f.argtypes = [
            ctypes.c_void_p,
            ctypes.c_char_p,
            ctypes.c_size_t,
            ctypes.c_uint,
        ]
        f.restype = ctypes.c_int
        if not f(
            device._Handle,
            fileDataCType,
            fileDataSizeCType,
            netidCType,
        ):
            raise RuntimeError("icsneoWriteMACsecConfig() failed!")

    # load extra functions that are on the DLL but not in python_ics
    print("Loading python_ics extensions from DLL...")
    if not hasattr(ics, "write_macsec_config"):
        ics.write_macsec_config = icsneoWriteMACsecConfig


def copy_array_to_ctypes(struct, k, array):
    bv = bytes(array)
    b = (ctypes.c_ubyte * len(bv)).from_buffer_copy(bv)
    setattr(struct, k, b)


def copy_dict_to_ctypes(struct, dict):
    for k, v in dict.items():
        if hasattr(struct, k):
            if issubclass(type(getattr(struct, k)), ctypes.Array):
                copy_array_to_ctypes(struct, k, v)
            else:
                setattr(struct, k, v)
        else:
            # print(f"WARNING: skipping entry. Unknown key '{k}'")
            pass


def argparse_raw(str):
    # Encode to our argparse parser that it shouldn't touch this
    ret = "R|\n"
    ret += str
    return ret


def serial_base36enc(ser_no):
    """
    Encode serial as base36 if needed and return the string representation of the serial number

    Args:
        ser_no:  Serial number integer
    """
    if int("AA0000", 36) < ser_no < int("ZZZZZZ", 36):
        return ics.base36enc(ser_no)
    else:
        return str(ser_no)  # Old devices don't do base36


def serial_base36dec(ser_no):
    """
    Decode serial as base36 if needed and return the integer representation of the serial number

    Args:
        ser_no:  Serial number string
    """
    serial36 = int(ser_no, 36)
    if int("AA0000", 36) < serial36 < int("ZZZZZZ", 36):
        return serial36
    else:
        return int(ser_no, 10)  # Old devices don't do base36


@contextmanager
def open_device(ser_no, tries=10, delay=1.0):
    """
    Context manager for a neovi device.  Opens the device, then auto-closes
    once the context manager falls out of scope

    Args:
        ser_no:  Serial number string of a connected device, will be validated

    Yields:
        an open device

    Examples:
        with open("GS0137") as device:
            ics_do_stuff(device)

    """
    device = None
    serial = serial_base36dec(ser_no)

    for i in range(tries):
        try:
            found = False
            # work around for supporting neovi server connections
            # ics.open_device will not work if already open in Vspy with server
            devices = ics.find_devices()
            for d in devices:
                if d.SerialNumber == serial:
                    device = ics.open_device(d)
                    found = True
                    break
            if found:
                # successfully opened
                break
            else:
                raise Exception(f"Could not find device to open {ser_no}")
        except Exception:
            device = None
            print(f"Failed to Open {ser_no}, Trying again... ({i+1}/{tries})")
            time.sleep(delay)

    if device is None:
        # could not find device and multiple retries
        devices = ics.find_devices()
        print("ERROR:  Device not found.  Known devices are:")
        print([serial_base36enc(dev.SerialNumber) for dev in devices])
        exit(ExitCode.EXIT_FAIL_HARDWARE_NOT_FOUND)

    try:
        yield device
    except Exception as e:
        print("ERROR: Open device succeeded, but yielding failed?")
        raise e
    finally:
        if device is not None:
            ics.close_device(device)


def get_hwnetid(name):
    """
    Get hardware network ID from string name in one of the following forms:
    COREMINI_NETWORK_ETHERNET, NETID_ETHERNET, or ETHERNET

    Args:
        name: Network name

    Returns:
        Hardware network ID
    """
    # NETID_XXX, use as is
    if name.startswith("NETID_"):
        if hasattr(ics, name):
            return getattr(ics, name)

    # COREMINI_NETWORK_XXX, convert
    if name.startswith("COREMINI_NETWORK_"):
        newname = "NETID_" + name[len("COREMINI_NETWORK_") :]
        if hasattr(ics, newname):
            return getattr(ics, newname)

    # assume no prefix, so try adding it
    newname = "NETID_" + name
    if hasattr(ics, newname):
        return getattr(ics, newname)

    raise AttributeError(f"Could not match network for {name}")


class MyArgParseHelpFormatter(argparse.ArgumentDefaultsHelpFormatter):
    def _split_lines(self, text, width):
        if text.startswith("R|"):
            lines = text.splitlines()[1:]
            return lines
        return super()._split_lines(text, width)


def dict_align_str(d, sort=False, nz=False):
    """
    Prints a dict nicely and key value pairs in aligned rows
    Primarily limited to simple "a": "b" dicts, might not look nice for complex value types

    Args:
        d: Some dictionary
        sort: Sort dictionary items
        nz: Excludes key value pairs with int value = 0

    Returns:
        A string that looks like this:
        aardvark: banana
        monkey:   apple
        ant:      pear
    """
    if not d:
        return ""
    strings = []
    longest_key_len = len(max(d.keys(), key=len))
    if sort:
        d = dict(sorted(d.items()))
    for k, v in d.items():
        key_len = len(k)
        diff = longest_key_len - key_len
        spaces = " " * diff
        string = f"{k}:{spaces} {v}"
        if nz:
            if not isinstance(v, int) or v > 0:
                strings.append(string)
        else:
            strings.append(string)
    ret = "\n".join(strings)
    return ret
