from util.util_eth import is_sfp
from util.util import (
    copy_array_to_ctypes,
    copy_dict_to_ctypes,
    get_hwnetid,
)
from util.util_sfp import (
    transmit_i2c_ICSSFP_CONFIG_MACSEC_MAP,
    transmit_i2c_ICSSFP_CONFIG_MACSEC_RULE,
    transmit_i2c_ICSSFP_CONFIG_MACSEC_SA,
    transmit_i2c_ICSSFP_CONFIG_MACSEC_SC,
    transmit_i2c_ICSSFP_CONFIG_MACSEC_SECY,
    transmit_i2c_ICSSFP_UPDATE_MACSEC,
)
import ics
import time
import yaml


def _macsec_load_to_sfp(device, netid, macsec_cfg):
    """
    Load a MACsec config to an SFP module from a dictionary
    """
    en = 0
    rx = 0
    rule = 0
    map = 0
    secy = 0
    sc = 0
    sa = 0
    nvm = 1
    clr = 0
    rst = 1
    # clear/reset any current macsec configuration
    sfp_ics_macsec_update(device, netid, rx, rule, map, secy, sc, sa, nvm, en, clr, rst)
    # allow time for sfp to write macsec configuration to phy over mdio
    time.sleep(1)
    # tx config
    if "tx" in macsec_cfg:
        en = 1
        rx = 0
        rst = 0
        if "rule" in macsec_cfg["tx"]:
            rule = 1
            # write macsec rule to sfp
            sfp_ics_macsec_send_rule(device, netid, macsec_cfg["tx"]["rule"])
        if "map" in macsec_cfg["tx"]:
            map = 1
            # write macsec map to sfp
            sfp_ics_macsec_send_map(device, netid, macsec_cfg["tx"]["map"])
        if "secy" in macsec_cfg["tx"]:
            secy = 1
            # write macsec secy to sfp
            sfp_ics_macsec_send_secy(device, netid, macsec_cfg["tx"]["secy"])
        if "sc" in macsec_cfg["tx"]:
            sc = 1
            # write macsec sc to sfp
            sfp_ics_macsec_send_sc(device, netid, macsec_cfg["tx"]["sc"])
        if "sa0" in macsec_cfg["tx"] and "sa1" in macsec_cfg["tx"]:
            sa = 1
            # write macsec sa0 to sfp
            sfp_ics_macsec_send_sa(device, netid, macsec_cfg["tx"]["sa0"])
            # push macsec sa config to phy
            sfp_ics_macsec_update(device, netid, rx, 0, 0, 0, 0, sa, 0, en, clr, rst)
            # allow time for sfp to write macsec configuration to phy over mdio
            time.sleep(0.5)
            # write macsec sa1 to sfp
            sfp_ics_macsec_send_sa(device, netid, macsec_cfg["tx"]["sa1"])
        # push entire macsec config to phy
        sfp_ics_macsec_update(
            device,
            netid,
            rx,
            rule,
            map,
            secy,
            sc,
            sa,
            nvm,
            en,
            clr,
            rst,
        )
        # allow time for sfp to write entire macsec configuration to phy over mdio
        time.sleep(1)
    # rx config
    if "rx" in macsec_cfg:
        en = 1
        rx = 1
        rst = 0
        if "rule" in macsec_cfg["rx"]:
            rule = 1
            # write macsec rule to sfp
            sfp_ics_macsec_send_rule(device, netid, macsec_cfg["rx"]["rule"])
        if "map" in macsec_cfg["rx"]:
            map = 1
            # write macsec map to sfp
            sfp_ics_macsec_send_map(device, netid, macsec_cfg["rx"]["map"])
        if "secy" in macsec_cfg["rx"]:
            secy = 1
            # write macsec secy to sfp
            sfp_ics_macsec_send_secy(device, netid, macsec_cfg["rx"]["secy"])
        if "sc" in macsec_cfg["rx"]:
            sc = 1
            # write macsec sc to sfp
            sfp_ics_macsec_send_sc(device, netid, macsec_cfg["rx"]["sc"])
        if "sa0" in macsec_cfg["rx"] and "sa1" in macsec_cfg["rx"]:
            sa = 1
            # write macsec sa0 to sfp
            sfp_ics_macsec_send_sa(device, netid, macsec_cfg["rx"]["sa0"])
            # push macsec sa config to phy
            sfp_ics_macsec_update(device, netid, rx, 0, 0, 0, 0, sa, 0, en, clr, rst)
            # allow time for sfp to write macsec configuration to phy over mdio
            time.sleep(0.5)
            # write macsec sa1 to sfp
            sfp_ics_macsec_send_sa(device, netid, macsec_cfg["rx"]["sa1"])
        # push entire macsec config to phy
        sfp_ics_macsec_update(
            device,
            netid,
            rx,
            rule,
            map,
            secy,
            sc,
            sa,
            nvm,
            en,
            clr,
            rst,
        )
        # allow time for sfp to write entire macsec configuration to phy over mdio
        time.sleep(1)


def _macsec_copy_struct(struct, dict):
    # copy identical items
    copy_dict_to_ctypes(struct, dict)
    if hasattr(struct, "rsvd") and "reserved" in dict:
        copy_array_to_ctypes(struct, "rsvd", dict["reserved"])


def _macsec_copy_rule(struct, rule):
    # copy identical items
    _macsec_copy_struct(struct, rule)
    # copy other items with different names
    copy_dict_to_ctypes(struct.key_vlantag_outer1, rule["key_outer1"]["vlanTag"])
    copy_dict_to_ctypes(struct.key_MPLS_outer1, rule["key_outer1"]["mpls"])
    copy_dict_to_ctypes(struct.mask_vlantag_outer1, rule["mask_outer1"]["vlanTag"])
    copy_dict_to_ctypes(struct.mask_MPLS_outer1, rule["mask_outer1"]["mpls"])
    copy_dict_to_ctypes(struct.key_vlantag_outer2, rule["key_outer2"]["vlanTag"])
    copy_dict_to_ctypes(struct.key_MPLS_outer2, rule["key_outer2"]["mpls"])
    copy_dict_to_ctypes(struct.mask_vlantag_outer2, rule["mask_outer2"]["vlanTag"])
    copy_dict_to_ctypes(struct.mask_MPLS_outer2, rule["mask_outer2"]["mpls"])


def _macsec_load_to_device(device, netid, macsec_cfg):
    """
    Load a MACsec config to the device from a dictionary
    """
    macsec = ics.structures.macsec_settings.macsec_settings()
    macsec.flags.nvm = 1  # save to non-volatile memory to load on boot

    # tx config
    if "tx" in macsec_cfg:
        if "rule" in macsec_cfg["tx"]:
            _macsec_copy_rule(macsec.tx.rule[0], macsec_cfg["tx"]["rule"])
        if "map" in macsec_cfg["tx"]:
            _macsec_copy_struct(macsec.tx.map[0], macsec_cfg["tx"]["map"])
        if "secy" in macsec_cfg["tx"]:
            _macsec_copy_struct(macsec.tx.secy[0], macsec_cfg["tx"]["secy"])
        if "sc" in macsec_cfg["tx"]:
            _macsec_copy_struct(macsec.tx.sc[0], macsec_cfg["tx"]["sc"])
        if "sa0" in macsec_cfg["tx"]:
            _macsec_copy_struct(macsec.tx.sa[0], macsec_cfg["tx"]["sa0"])
        if "sa1" in macsec_cfg["tx"]:
            _macsec_copy_struct(macsec.tx.sa[1], macsec_cfg["tx"]["sa1"])
        macsec.tx.flags.en = 1
        macsec.flags.en = 1

    # rx config
    if "rx" in macsec_cfg:
        if "rule" in macsec_cfg["rx"]:
            _macsec_copy_rule(macsec.rx.rule[0], macsec_cfg["rx"]["rule"])
        if "map" in macsec_cfg["rx"]:
            _macsec_copy_struct(macsec.rx.map[0], macsec_cfg["rx"]["map"])
        if "secy" in macsec_cfg["rx"]:
            _macsec_copy_struct(macsec.rx.secy[0], macsec_cfg["rx"]["secy"])
        if "sc" in macsec_cfg["rx"]:
            _macsec_copy_struct(macsec.rx.sc[0], macsec_cfg["rx"]["sc"])
        if "sa0" in macsec_cfg["rx"]:
            _macsec_copy_struct(macsec.rx.sa[0], macsec_cfg["rx"]["sa0"])
        if "sa1" in macsec_cfg["rx"]:
            _macsec_copy_struct(macsec.rx.sa[1], macsec_cfg["rx"]["sa1"])
        macsec.rx.flags.en = 1
        macsec.flags.en = 1

    # send structure to the device
    macsec_bytes = bytearray(macsec)
    assert len(macsec_bytes) == 2040
    ics.write_macsec_config(device, macsec_bytes, netid)


def macsec_load(device, config_netid_name, port_netid_name, macsec_cfg):
    """
    Load a MACsec config to the device from a dictionary
    """
    if is_sfp(
        config_netid_name
    ):  # ics sfp module - load over i2c with ics config subcommands
        print(f"Loading MACsec config to SFP {config_netid_name}...")
        netid = get_hwnetid(config_netid_name)
        _macsec_load_to_sfp(device, netid, macsec_cfg)
    else:  # normal device - load via MACSEC_SETTINGS
        print(f"Loading MACsec config to Port {port_netid_name}...")
        netid = get_hwnetid(port_netid_name)
        _macsec_load_to_device(device, netid, macsec_cfg)


def macsec_load_from_yaml(device, config_netid_name, port_netid_name, yml):
    """
    Load a MACsec config to the device from a yaml file
    """
    with open("yaml/" + yml, "r") as file:
        macsec_cfg = yaml.safe_load(file)  # read macsec configuration from yaml
        macsec_load(device, config_netid_name, port_netid_name, macsec_cfg)
        # print macsec confirmation
        print(
            f"Successfully configured MACsec on {config_netid_name}/{port_netid_name}"
        )


def macsec_clear(device, config_netid_name, port_netid_name):
    """
    Clear out the MACsec configuration by loading an empty config
    """
    macsec_cfg = {}
    macsec_load(device, config_netid_name, port_netid_name, macsec_cfg)
    # print macsec confirmation
    print(f"Cleared MACsec on {port_netid_name}")


def sfp_ics_macsec_send_rule(device, netid, rule):
    data = []
    data.append(rule["index"])
    for x in range(6):
        data.append(rule["key_MAC_DA"][x])
    for x in range(6):
        data.append(rule["mask_MAC_DA"][x])
    for x in range(6):
        data.append(rule["key_MAC_SA"][x])
    for x in range(6):
        data.append(rule["mask_MAC_SA"][x])
    for x in bytearray(rule["key_Ethertype"].to_bytes(2, "little")):
        data.append(x)
    for x in bytearray(rule["mask_Ethertype"].to_bytes(2, "little")):
        data.append(x)
    for x in bytearray(rule["key_outer1"]["vlanTag"]["VID"].to_bytes(2, "little")):
        data.append(x)
    data.append(rule["key_outer1"]["vlanTag"]["PRI_CFI"])
    for x in bytearray(rule["key_outer1"]["mpls"]["MPLS_label"].to_bytes(4, "little")):
        data.append(x)
    data.append(rule["key_outer1"]["mpls"]["exp"])
    for x in bytearray(rule["mask_outer1"]["vlanTag"]["VID"].to_bytes(2, "little")):
        data.append(x)
    data.append(rule["mask_outer1"]["vlanTag"]["PRI_CFI"])
    for x in bytearray(rule["mask_outer1"]["mpls"]["MPLS_label"].to_bytes(4, "little")):
        data.append(x)
    data.append(rule["mask_outer1"]["mpls"]["exp"])
    for x in bytearray(rule["key_outer2"]["vlanTag"]["VID"].to_bytes(2, "little")):
        data.append(x)
    data.append(rule["key_outer2"]["vlanTag"]["PRI_CFI"])
    for x in bytearray(rule["key_outer2"]["mpls"]["MPLS_label"].to_bytes(4, "little")):
        data.append(x)
    data.append(rule["key_outer2"]["mpls"]["exp"])
    for x in bytearray(rule["mask_outer2"]["vlanTag"]["VID"].to_bytes(2, "little")):
        data.append(x)
    data.append(rule["mask_outer2"]["vlanTag"]["PRI_CFI"])
    for x in bytearray(rule["mask_outer2"]["mpls"]["MPLS_label"].to_bytes(4, "little")):
        data.append(x)
    data.append(rule["mask_outer2"]["mpls"]["exp"])
    for x in bytearray(rule["key_bonus_data"].to_bytes(2, "little")):
        data.append(x)
    for x in bytearray(rule["mask_bonus_data"].to_bytes(2, "little")):
        data.append(x)
    data.append(rule["key_tag_match_bitmap"])
    data.append(rule["mask_tag_match_bitmap"])
    data.append(rule["key_packet_type"])
    data.append(rule["mask_packet_type"])
    for x in bytearray(rule["key_inner_vlan_type"].to_bytes(2, "little")):
        data.append(x)
    for x in bytearray(rule["mask_inner_vlan_type"].to_bytes(2, "little")):
        data.append(x)
    for x in bytearray(rule["key_outer_vlan_type"].to_bytes(2, "little")):
        data.append(x)
    for x in bytearray(rule["mask_outer_vlan_type"].to_bytes(2, "little")):
        data.append(x)
    data.append(rule["key_num_tags"])
    data.append(rule["mask_num_tags"])
    data.append(rule["key_express"])
    data.append(rule["mask_express"])
    for x in bytearray(rule["isMPLS"].to_bytes(1, "little")):
        data.append(x)
    for x in range(5):
        data.append(rule["reserved"][x])
    for x in bytearray(rule["enable"].to_bytes(1, "little")):
        data.append(x)
    transmit_i2c_ICSSFP_CONFIG_MACSEC_RULE(device, netid, len(data), data)


def sfp_ics_macsec_send_map(device, netid, map):
    data = []
    data.append(map["index"])
    for x in bytearray(map["sectag_sci"].to_bytes(8, "little")):
        data.append(x)
    data.append(map["secYIndex"])
    for x in bytearray(map["isControlPacket"].to_bytes(1, "little")):
        data.append(x)
    data.append(map["scIndex"])
    for x in bytearray(map["auxiliary_plcy"].to_bytes(1, "little")):
        data.append(x)
    data.append(map["ruleId"])
    for x in range(5):
        data.append(map["reserved"][x])
    for x in bytearray(map["enable"].to_bytes(1, "little")):
        data.append(x)
    transmit_i2c_ICSSFP_CONFIG_MACSEC_MAP(device, netid, len(data), data)


def sfp_ics_macsec_send_secy(device, netid, secy):
    data = []
    data.append(secy["index"])
    for x in bytearray(secy["controlled_port_enabled"].to_bytes(1, "little")):
        data.append(x)
    data.append(secy["validate_frames"])
    data.append(secy["strip_sectag_icv"])
    data.append(secy["cipher"])
    data.append(secy["confidential_offset"])
    for x in bytearray(secy["icv_includes_da_sa"].to_bytes(1, "little")):
        data.append(x)
    for x in bytearray(secy["replay_protect"].to_bytes(1, "little")):
        data.append(x)
    for x in bytearray(secy["replay_window"].to_bytes(4, "little")):
        data.append(x)
    for x in bytearray(secy["protect_frames"].to_bytes(1, "little")):
        data.append(x)
    data.append(secy["sectag_offset"])
    data.append(secy["sectag_tci"])
    for x in bytearray(secy["mtu"].to_bytes(2, "little")):
        data.append(x)
    for x in range(6):
        data.append(secy["reserved"][x])
    for x in bytearray(secy["enable"].to_bytes(1, "little")):
        data.append(x)
    transmit_i2c_ICSSFP_CONFIG_MACSEC_SECY(device, netid, len(data), data)


def sfp_ics_macsec_send_sc(device, netid, sc):
    data = []
    data.append(sc["index"])
    data.append(sc["secYIndex"])
    for x in bytearray(sc["sci"].to_bytes(8, "little")):
        data.append(x)
    data.append(sc["sa_index0"])
    data.append(sc["sa_index1"])
    for x in bytearray(sc["sa_index0_in_use"].to_bytes(1, "little")):
        data.append(x)
    for x in bytearray(sc["sa_index1_in_use"].to_bytes(1, "little")):
        data.append(x)
    for x in bytearray(sc["enable_auto_rekey"].to_bytes(1, "little")):
        data.append(x)
    for x in bytearray(sc["isActiveSA1"].to_bytes(1, "little")):
        data.append(x)
    for x in range(7):
        data.append(sc["reserved"][x])
    for x in bytearray(sc["enable"].to_bytes(1, "little")):
        data.append(x)
    transmit_i2c_ICSSFP_CONFIG_MACSEC_SC(device, netid, len(data), data)


def sfp_ics_macsec_send_sa(device, netid, sa):
    data = []
    data.append(sa["index"])
    for x in range(32):
        data.append(sa["sak"][x])
    for x in range(16):
        data.append(sa["hashKey"][x])
    for x in range(12):
        data.append(sa["salt"][x])
    for x in bytearray(sa["ssci"].to_bytes(4, "little")):
        data.append(x)
    data.append(sa["AN"])
    for x in bytearray(sa["nextPN"].to_bytes(8, "little")):
        data.append(x)
    for x in range(5):
        data.append(sa["reserved"][x])
    for x in bytearray(sa["enable"].to_bytes(1, "little")):
        data.append(x)
    transmit_i2c_ICSSFP_CONFIG_MACSEC_SA(device, netid, len(data), data)


def sfp_ics_macsec_update(
    device, netid, rx, rule, map, secy, sc, sa, nvm, en, clr, rst
):
    data = []
    byte0 = rule << 0
    byte0 |= map << 1
    byte0 |= secy << 2
    byte0 |= sc << 3
    byte0 |= sa << 4
    byte0 |= rx << 5
    byte0 |= nvm << 6
    byte0 |= en << 7
    data.append(byte0)
    byte1 = clr << 0
    byte1 |= rst << 1
    data.append(byte1)
    transmit_i2c_ICSSFP_UPDATE_MACSEC(device, netid, data)
